  ' -----------------------------------------------------------
  ' PAJ7620U2 Gesture Rec.bas
  ' A program for Micromite to communicate with the CJMCU-7620
  ' Gesture Recognition module (based on the PixArt PAJ7620U2 chip)
  ' for hand gesture sensing and recognition.
  '
  ' Written by Jim Rowe (with help from Tim Blythman) for Silicon Chip.
  ' Last revision 1/9/2021 at 11:20 am
  '
  ' Notes:
  ' 1. Communication with the module is via the Micromite's
  ' I2C port, which uses pin 17 for SCL and pin 18 for SDA.
  ' 2. The PAJ7620U2 chip's 7-bit I2C address is specified as 73h.
  ' (MMBasic's I2C function expects a 7-bit I2C address since it adds
  ' the lsb read/write bit automatically; R = 1, W = 0).
  ' 3. When a gesture is recognised by the PAJ7620 it stores the
  ' recognition code for the gesture in Bank0 addresses 43h & 44h.
  ' 4. Address EFh in Bank0 and Bank1 of the PAJ7620 is the
  ' Bank Select address (00 = Bank0, 01h = Bank1)
  ' 5. The PAJ7620 chip's ID information is stored in Bank0 addresses
  ' 00, 01 and 02h, with address 01 holding 76h, address 00 holding 20h
  ' and address 02 holding the version number (usually 01h). These
  ' addresses are read-only, and used by the program to check if
  ' the PAJ7620 chip is present and active.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  
  DIM INTEGER PAJinit(216) ' array to store chip init reg addresses & data
  SetPAJRegs          ' load the register init data into the array elements
  DIM INTEGER IDbytes(2)  ' array to hold PAJ7620 chip ID bytes 0-2
  DIM GString$ = "                  " ' blank until gesture is recognised
  DIM IntFlg = 0          ' screen touched flag (ignored here)
  DIM IDFlg = 0           ' flag for device ID (1 = not found)
  DIM INTEGER Cntr        ' loop counter
  DIM INTEGER RegAdnData  ' 16-bit data stored in PAJinit array
  DIM INTEGER AddrByte    ' upper 8 bits = address
  DIM INTEGER Idata       ' lower 8 bits = initialising data
  DIM INTEGER GByte0      ' to store gesture flag register 43h contents
  DIM INTEGER GByte1      ' to store gesture flag register 44h contact
  I2C OPEN 100, 200       ' enable the Micromite I2C module in master mode
                          ' (speed = 100kHz, timeout = 200 milliseconds
  
  ' ---------------------------------------------------------------------
  ' start of main part of program
  PAUSE 1  ' wait 1ms for PAJ7620 to stabilise after power up
  ShowStUpScrn  ' now show the startup screen
  PAJ7620Init   ' then initialise the PAJ7620 device in module
  PRINT "PAJ7620 initialised..."
  PAUSE 2000
  PRINT "Start gesture input:"     ' invite user to input gestures
  ShowOpScrn    ' and show operating screen
  
  ' main program loop starts here
  DO
    ' first read PAJ7620 register 43h in Bank0 (Gesture flag register)
    ReadGReg0
    IF (GByte0 <> 0) THEN ' if GByte0 is other than zero
      GoReadGesture   ' go decode GByte0 & find GString$
      TEXT 10, MM.VRes*3/8, GString$, LM, 1,2, White,Black  ' display
      PRINT GString$    ' print as well on PC serial monitor screen
      PAUSE 1000        ' pause to allow it to be seen
    ELSE                ' but if GByte = 0
      GString$ = "                  "
      TEXT 10, MM.VRES*3/8, GString$, LM, 1,2, WHITE,BLACK 'blank display
    ENDIF
  LOOP
' end of main program, subroutines follow
' **********************************************************************

' subroutine to see if PAJ7620 is present & functioning, get its ID
SUB GoCheckChip
  I2C WRITE &H73,0,1,0    ' first write first address to read from (00)
  I2C READ &H73,0,3, IDbytes()  ' then get the three ID bytes
  PRINT "Device ID is: ", HEX$(IDbytes(1),2), HEX$(IDbytes(0),2)," V." HEX$(IDbytes(2),2)
END SUB
' -----------------------------------------------------------------
  ' subroutine to set IntFlag when screen is touched (ignored here)
SUB TchInt
  IntFlg = 1
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display the startup screen
SUB ShowStUpScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, " SILICON CHIP's", LM, 1,2, Green, Black
  TEXT 10, MM.VRes/4, "Micromite Program", LM, 1,2, Green, Black
  TEXT 10, MM.VRes*3/8, "for trying out the", LM, 1,2, Green, Black
  TEXT 10, MM.VRes/2, "CJMCU Hand Gesture", LM, 1,2, Green, Black
  TEXT 10, MM.VRes*5/8, "recognition module", LM, 1,2, Green, Black  
END SUB
  ' -----------------------------------------------------------------
    ' subroutine to initialise the PAJ7620
SUB PAJ7620Init
  I2C WRITE &H73,0,2,&HEF,&H00  ' send 00 to register EFh
  I2C WRITE &H73,0,2,&HEF,&H00  ' then send again, to select Bank0
  GoCheckChip              ' now go and read Bank0 addresses 00,01,02
  IF IDbytes(0) <> &H20 THEN IDFlg = 1
  IF IDbytes(1) <> &H76 THEN IDFlg = 1
  IF IDFlg = 1 THEN PRINT "Error: PAJ7620 not found!"
  FOR Cntr = 0 TO 215 STEP 1
    RegAdnData = PAJinit(Cntr)              ' fetch next PAJinit() 16-bit data
    AddrByte = (RegAdnData AND &HFF00) >> 8 ' separate address byte
    IData = (RegAdnData AND &H00FF)         ' and data byte
    I2C WRITE &H73,0,2, AddrByte, IData     ' then send them to the PAJ7620
  NEXT
  I2C WRITE &H73,0,2,&HEF,&H00  ' send 00h to reg EFh, to return to Bank0
  I2C WRITE &H73,0,2,&H03,&H01  ' & send 01h to reg 03h, to wake up PAJ7620 <**
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display the operating screen
SUB ShowOpScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, "Gesture Sensed:", LM, 1,2, RED, Black
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read Gesture register 0 (Bank0, 43h)
SUB ReadGReg0
  I2C Write &H73,0,1,&H43   ' write register address so we
  I2C READ &H73,0,1, GByte0 ' can read data from it, then
  GByte0 = (GByte0 AND &HFF)  ' strip it down to a single byte
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read Gesture register 1 (Bank0, 44h)
SUB ReadGReg1
  I2C Write &H73,0,1,&H44   ' write register address so we
  I2C READ &H73,0,1, GByte1 ' can read data from it, then
  GByte1 = (GByte1 AND &H01)  ' strip it down to least signif bit
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read the gesture recognised
SUB GoReadGesture
  SELECT CASE GByte0
    CASE &H02   ' 02h in Reg 43h = Right Flag
      PAUSE 500
      ReadGReg0   ' read reg 43h again
      SELECT CASE GByte0
        CASE &H10
          GString$ = "Forward           "
        CASE &H20
          GString$ = "Backward          "
        CASE ELSE
          GString$ = "Right             "
      END SELECT
    CASE &H01   ' 01h in Reg 43h = Left Flag
      PAUSE 500
      ReadGReg0   ' read reg 43h again
      SELECT CASE GByte0
        CASE &H10
          GString$ = "Forward           "
        CASE &H20
          GString$ = "Backward          "
        CASE ELSE
          GString$ = "Left              "
      END SELECT
    CASE &H08     ' 08h in Reg 43h = Up Flag  
      PAUSE 500
      ReadGReg0   ' read reg 43h again
      SELECT CASE GByte0
        CASE &H10
          GString$ = "Forward           "
        CASE &H20
          GString$ = "Backward          "
        CASE ELSE
          GString$ = "Up                "
      END SELECT
    CASE &H04   ' 04h in Reg 43h = Down Flag
      PAUSE 500
      ReadGReg0   ' read reg 43h again
      SELECT CASE GByte0
        CASE &H10
          GString$ = "Forward           "
        CASE &H20
          GString$ = "Backward          "
        CASE ELSE
          GString$ = "Down              "
      END SELECT
    CASE &H10   ' 10h in Reg 43h = Forward Flag
      GString$ = "Forward           "
    CASE &H20   ' 20h in Reg 43h = Backward Flag
      GString$ = "Backward          "
    CASE &H40   ' 40h in Reg 43h = Clockwise Flag
      GString$ = "Clockwise         "
    CASE &H80   ' 80h in Reg 43h = Counterclockwise Flag
      GString$ = "Counter Clockwise "
    CASE ELSE
      ReadGReg1   ' now go read Reg 44h
      IF GByte1 = &H01 THEN GString$ = "Wave            "
    END SELECT
END SUB

' ------------------------------------------------------------------
' subroutine to setup the PAJ7620 registers. Register names and default
' data values taken from the PixArt datasheet for the PAJ7620.
SUB SetPAJRegs
PAJinit(0) = &HEF00   ' send 00h TO ADDR EFh, to switch to Bank 0
PAJinit(1) = &H3229   ' addr 32h = R_CursorUseTop/BGModel/InvertY/InvertX
PAJinit(2) = &H3301   ' addr 33h = R_PositionFilterStartSizeTh(7:0)
PAJinit(3) = &H3400   ' addr 34h = R_PositionFilterStartSizeTh(8)
PAJinit(4) = &H3501   ' addr 35h = R_ProcessFilterStartSizeTh(7:0)
PAJinit(5) = &H3600   ' addr 36h = R_ProcessFilterStartSizeTh(8)
PAJinit(6) = &H3707   ' addr 37h = R_CursorClampLeft(4:0)
PAJinit(7) = &H3817   ' addr 38h = R_CursorClampRight(4:0)
PAJinit(8) = &H3906   ' addr 39h = R_CursorClampUp(4:0)
PAJinit(9) = &H3A12   ' addr 3Ah = R_CursorClampDown(4:0)
PAJinit(10) = &H3F00  ' addr 3Fh = R_oscrcpd default 00h
PAJinit(11) = &H4002  ' addr 40h = R_MCU_IntFlagGClr/Inv
PAJinit(12) = &H41FF  ' addr 41h = R_Int_1_En(7:0) default FFh
PAJinit(13) = &H4201  ' addr 42h = R_Int_2_En(7:0) default FFh
PAJinit(14) = &H462D  ' addr 46h = R_AELedOff_UB(7:0) default 60h
PAJinit(15) = &H470F  ' addr 47h = R_AELedOff_LB(7:0) default 20h
PAJinit(16) = &H483C  ' addr 48h = R_AE_Exposure_UB(7:0) default 20h
PAJinit(17) = &H4900  ' addr 49h = R_AE_Exposure_UB(15:8) default 03h
PAJinit(18) = &H4A1E  ' addr 4Ah = R_AE_Exposure_LB(7:0) default C8h
PAJinit(19) = &H4B00  ' addr 4Bh = R_AE_Exposure_LB(15:8) default 00h
PAJinit(20) = &H4C20  ' addr 4Ch = R_AE_Gain_UB(7:0) default 14h
PAJinit(21) = &H4D00  ' addr 4Dh = R_AE_Gain_LB(7:0) default 00h
PAJinit(22) = &H4E1A  ' addr 4Eh = R_SleepAEAG_AutoDisable(4) default 01h
PAJinit(23) = &H4F14  ' addr 4Fh = R_AE_Gain_Default(6:0)  default 14h
PAJinit(24) = &H5000  ' addr 50h = R_Exp_Sel(2:0) default 00h
PAJinit(25) = &H5110  ' addr 51h = R_Manual_GG(0)/R_Manual_Exposure(1) etc...
PAJinit(26) = &H5200  ' addr 52h = R_SleepAE1/AE2/AG1/AG2  default 00h
PAJinit(27) = &H5C02  ' addr 5Ch = R_SenClkPrd(5:0) default 02h
PAJinit(28) = &H5D00  ' addr 5Dh = R_SENCLK_Control  default 00h
PAJinit(29) = &H5E10  ' addr 5Eh = R_Other_CLK_manual  default 00h
PAJinit(30) = &H5F3F  ' addr 5Fh = R_CLK_manual default 7Fh
PAJinit(31) = &H6128  ' addr 61h = R_osc_code(6:0) default 34h
PAJinit(32) = &H6303  ' addr 63h = R_Ref_CLK_Cnt_UB(15:8) default 03h
PAJinit(33) = &H64F7  ' addr 64h = R_Ref_CLK_Cnt_UB(7:0) default F7h
PAJinit(34) = &H6503  ' addr 65h = R_Ref_CLK_Cnt_LB(15:8) default 03h
PAJinit(35) = &H66D9  ' addr 66h = R_Ref_CLK_Cnt_LB(7:0) default D9h
PAJinit(36) = &H6703  ' addr 67h = R_OSC_Wait(7:0) default 03h
PAJinit(37) = &H6801  ' addr 68h = T_osc_range(1:0) default 01h
PAJinit(38) = &H69C8  ' addr 69h = R_Pox_UB(7:0) default C8h
PAJinit(39) = &H6A40  ' addr 6Ah = R_Pox_LB(7:0) default 40h
PAJinit(40) = &H6D04  ' addr 6Dh = R_SRAM_DS(3:0) default 04h (RESVD REGS!)
PAJinit(41) = &H6E00  ' addr 6Eh = R_BIST_mode(3:0) default 00h (ditto!)
PAJinit(42) = &H6F00  ' addr 6Fh = R_BIST_mode_1(3:0) default 00h (ditto!)
PAJinit(43) = &H7080  ' addr 70h = R_EFuse_A(5:0)/R_EFuse_READ/R_EFuse_ENB
PAJinit(44) = &H7100  ' addr 71h = R_EFuse_SEL(2:0)/R_PROGRAM_BitsCNT(3:0) def 00h
PAJinit(45) = &H7200  ' addr 72h = R_EFuse_PROGRAM_En/EFuse_PROGRAM default 00h
PAJinit(46) = &H7300  ' addr 73h = R_EFuse_PROGRAM_CLK Cnt(14:8) default 00h
PAJinit(47) = &H74F0  ' addr 74h = R_EFuse_PROGRAM_CLK Cnt(7:0) default F0h
PAJinit(48) = &H7500  ' addr 75h = R_EFuse_PROGRAM_Data (7:0) default 00h
PAJinit(49) = &H8042  ' addr 80h = Im_GPIO3
PAJinit(50) = &H8144  ' addr 81h = Tm_GPIO3_OEL/Tm_GPIO3_IEB/R_GPIO3  default 01h
PAJinit(51) = &H8204  ' addr 82h = Im_INT/Tm_INT_OEL/Tm_INT_IEB/Tm_INT default 00/01h
PAJinit(52) = &H8320  ' addr 83h = R_LightThd(7:0) default 20h
PAJinit(53) = &H8420  ' addr 84h = R_GestureStartTh(7:0) default 20h
PAJinit(54) = &H8500  ' addr 85h = R_GestureStartTh(9:8) default 00h
PAJinit(55) = &H8610  ' addr 86h = R_GestureEndTh(7:0) default 10h
PAJinit(56) = &H8700  ' addr 87h = R_GestureEndTh(9:8) default 00h
PAJinit(57) = &H8805  ' addr 88h = R_ObjectMinZ(4:0) default 05h
PAJinit(58) = &H8918  ' addr 89h = R_ObjectMaxZ(5:0) default 18h
PAJinit(59) = &H8A10  ' addr 8Ah = R_LockReg/R_AutoLock default 00/10h (RESVD REG!)
PAJinit(60) = &H8B01  ' addr 8Bh = R_Cursor_ObjectSizeTh(7:0) default 10h
PAJinit(61) = &H8C37  ' addr 8Ch = R_PositionResolution(2:0) default 07h
PAJinit(62) = &H8D00  ' addr 8Dh = R_TimeDelayNum (7:0) default 00h
PAJinit(63) = &H8EF0  ' addr 8Eh = R_Disable45Degree/R_45DegreeRatio(3:0) def 00/F0h
PAJinit(64) = &H8F81  ' addr 8Fh = R_XtoYGain/R_XYGainRatio(3:0) default 01/08h
PAJinit(65) = &H9006  ' addr 90h = R_NoMotionCountThd(6:0) default 0Ch
PAJinit(66) = &H9106  ' addr 91h = R_NoObjectCountTHd(6:0) default 06h
PAJinit(67) = &H921E  ' addr 92h = R_NormalisedImageWidth(4:0) default 1Eh
PAJinit(68) = &H930D  ' addr 93h = R_XDirectionThd(4:0) default 0Dh
PAJinit(69) = &H940A  ' addr 94h = R_YDirectionThd(4:0) default 0Ah
PAJinit(70) = &H950A  ' addr 95h = R_ZDirectionThd(4:0) default 0Ch
PAJinit(71) = &H960C  ' addr 96h = R_ZDirectionXYThd(4:0) default 0Ah
PAJinit(72) = &H9705  ' addr 97h = R_ZDirectionAngleThd(3:0) default 04h
PAJinit(73) = &H980A  ' addr 98h = R_RotateAngleThd(5:0) default 0Ah
PAJinit(74) = &H9941  ' addr 99h = R_RotateContiEnh/R_RotateContiThd(5:0) def 01/00h
PAJinit(75) = &H9A14  ' addr 9Ah = R_RotateXYThd(4:0) default 0Ah
PAJinit(76) = &H9B0A  ' addr 9Bh = R_RotateZThd(4:0) default 0Ah
PAJinit(77) = &H9C3F  ' addr 9Ch = R_FilterWeight(1:0)/R_FilterDistThd(4:0) def 03/0Ah
PAJinit(78) = &H9D33  ' addr 9Dh = R_StartDistThd(3:0)/R_EndDistThd(3:0) default 03h
PAJinit(79) = &H9FF9  ' addr 9Fh = R_RotateEnh/ZDirectionEnh/YDirectionEnh/XDirEnh def 01h
PAJinit(80) = &HA048  ' addr A0h = R_BGUpdateMaxIntensity(7:0) default 03h
PAJinit(81) = &HA113  ' addr A1h = R_BGFilterLengthUp(1:0)/BGFilterLengthDown(1:0)
PAJinit(82) = &HA210  ' addr A2h = R_BGDiffThd(5:0)  default 08h
PAJinit(83) = &HA308  ' addr A3h = R_BGUpdateFreq(7:0) default 00h
PAJinit(84) = &HA430  ' addr A4h = R_BGUpdateFreq(9:8)/R_BGResolution(1:0) default 03h
PAJinit(85) = &HA519  ' addr A5h = R_FilterImage/FilterAverage_Mode/R_UseLightWeight def 01/00h
PAJinit(86) = &HA904  ' addr A9h = R_DiffAngleThd(3:0)  default 04h
PAJinit(87) = &HAA1E  ' addr AAh = R_ImageHeight(5:0) default 1Eh
PAJinit(88) = &HAB1E  ' addr ABh = R_ImageWidth(5:0) default 1Eh
PAJinit(89) = &HCC19  ' addr CCh = R_YtoZSum(5:0) default 1Ah
PAJinit(90) = &HCD0B  ' addr CDh = R_YtoZFactor(5:0) default 0Dh
PAJinit(91) = &HCE13  ' addr CEh = R_PositionFilterLength/ProcessFilterLength(2:0) def 03h
PAJinit(92) = &HCF64  ' addr CFh = R_WaveCountThd(3:0)/WaveAngleThd(3:0) default 03/06h
PAJinit(93) = &HD021  ' addr D0h = R_AbortCountThd(2:0)/AbortXYRatio(4:0) default 22h
PAJinit(94) = &HD10F ' addr D1h = R_AbortLength(6:0) default 0Fh
PAJinit(95) = &HD288 ' addr D2h = R_AbortIntervalCountThd/ConfirmMode/WaveEnh default 88h
PAJinit(96) = &HE001 ' addr E0h = R_DSP_TestObj_Size/R_DSP_Test_Mode (3:0) default 01/00h
PAJinit(97) = &HE104 ' addr E1h = R_DSP_TestObj_Hi(7:0) default 04h
PAJinit(98) = &HE241 ' addr E2h = R_DSP_TestObj_Lo(7:0) default 41h
PAJinit(99) = &HE3D6 ' addr E3h = R_DSP_TestObj_Dist(3:0)/DSP_TestObj_Speed(3.0 def 06/0Dh
PAJinit(100) = &HE400 ' addr E4h = R_DSP_TestObj_PosL(5:0) default 00h
PAJinit(101) = &HE50C ' addr E5h = R_DSP_TestObj_PosR(5:0) default 8Ch
PAJinit(102) = &HE60A ' addr E6h = R_DSP_TestObj_PosU(5:0) default 4Ah
PAJinit(103) = &HE700 ' addr E7h = R_DSP_TestObj_PosD(5:0) default 00h
PAJinit(104) = &HE800 ' addr E8h = R_DSP_TestObj_FrameCnt/DSP_Test_Noise_Enh default 00h
PAJinit(105) = &HE900 ' addr E9h = R_DSP_Test_Noise_Enh default 00h
PAJinit(106) = &HEE07 ' addr EEh = R_RegBank Reset/TGRegBank_SWRstn/DMSRegBank_SWRstn def 01h
PAJinit(107) = &HEF01 ' send 01h TO ADDR EFh, to SWITCH TO BANK 1 <<*********
PAJinit(108) = &H001E ' addr 00h = Cmd_HSize (5:0) default 1Eh (= 30 dec)
PAJinit(109) = &H011E ' addr 01h = Cmd_VSize (5:0) default 1Eh (= 30 dec)
PAJinit(110) = &H020F ' addr 02h = Cmd_HStart (5:0) default 00h
PAJinit(111) = &H0310 ' addr 03h = Cmd_VStart (5:0) default 00h
PAJinit(112) = &H0402 ' addr 04h = Cmd_ASkip_V/ASkip_H/DAvg_V/VFlip/HFlip default 01/00h
PAJinit(113) = &H0500 ' addr 05h = Cmd_DebugMode default 00h
PAJinit(114) = &H06B0 ' addr 06h = Cmd_CDS_Timing_Control default 4B0h
PAJinit(115) = &H0704 ' addr 07h = Cmd_CDS_Timing_Control  "
PAJinit(116) = &H080D ' addr 08h = Cmd_CDS_Timing_Control default 0Dh
PAJinit(117) = &H090E ' addr 09h = Cmd_CDS_Timing_Control default 0Eh
PAJinit(118) = &H0A9C ' addr 0Ah = Cmd_CDS_Timing_Control default 708h
PAJinit(119) = &H0B04 ' addr 0Bh = Cmd_CDS_Timing_Control  "
PAJinit(120) = &H0C05 ' addr 0Ch = Cmd_CDS_Timing_Control default 05h
PAJinit(121) = &H0D0F ' addr 0Dh = Cmd_CDS_Timing_Control default 0Fh
PAJinit(122) = &H0E02 ' addr 0Eh = Cmd_CDS_Timing_Control default 02h
PAJinit(123) = &H0F12 ' addr 0Fh = Cmd_CDS_Timing_Control default 12h
PAJinit(124) = &H1002 ' addr 10h = Cmd_CDS_Timing_Control default 02h
PAJinit(125) = &H1102 ' addr 11h = Cmd_CDS_Timing_Control default 02h
PAJinit(126) = &H1200 ' addr 12h = Cmd_CDS_Timing_Control default 00h
PAJinit(127) = &H1301 ' addr 13h = Cmd_CDS_Timing_Control default 01h
PAJinit(128) = &H1405 ' addr 14h = Cmd_CDS_Timing_Control default 05h
PAJinit(129) = &H1507 ' addr 15h = Cmd_CDS_Timing_Control default 07h
PAJinit(130) = &H1605 ' addr 16h = Cmd_CDS_Timing_Control default 05h
PAJinit(131) = &H1707 ' addr 17h = Cmd_CDS_Timing_Control default 07h
PAJinit(132) = &H1801 ' addr 18h = Cmd_CDS_Timing_Control default 01h
PAJinit(133) = &H1904 ' addr 19h = Cmd_CDS_Timing_Control default 04h
PAJinit(134) = &H1A05 ' addr 1Ah = Cmd_CDS_Timing_Control default 05h
PAJinit(135) = &H1B0C ' addr 1Bh = Cmd_CDS_Timing_Control default 0Ch
PAJinit(136) = &H1C2A ' addr 1Ch = Cmd_CDS_Timing_Control default 2Ah
PAJinit(137) = &H1D01 ' addr 1Dh = Cmd_CDS_Timing_Control default 01h
PAJinit(138) = &H1E00 ' addr 1Eh = R_TG_Mode  default 00
PAJinit(139) = &H2100 ' addr 21h = R_OnOffNegComp(7:0) default 00h
PAJinit(140) = &H2200 ' addr 22h = R_OnOffNegComp(8) default 00h
PAJinit(141) = &H2300 ' addr 23h = R_RowDummyTrailingO_EnH default 00h
PAJinit(142) = &H2501 ' addr 25h = R_LensShadingComp_EnH  default 00h
PAJinit(143) = &H2600 ' addr 26h = R_OffsetX (6:0) default 00h
PAJinit(144) = &H2739 ' addr 27h = R_OffsetY (6:0) default 00h
PAJinit(145) = &H287F ' addr 28h = R_LSC (6:0) default 40h
PAJinit(146) = &H2908 ' addr 29h = R_LSFT (3:0) default 0Ah
PAJinit(147) = &H3003 ' addr 30h = R_LED_SoftStart_time (7:0) default 03h
PAJinit(148) = &H3100 ' addr 31h = R_LED2_DAC_EnL/LED1_DAC_EnL etc default 00h
PAJinit(149) = &H321A ' addr 32h = R_LED1_DAC_UB (4:0) default 14h
PAJinit(150) = &H331A ' addr 33h = R_LED2_DAC_UB (4:0) default 14h
PAJinit(151) = &H3407 ' addr 34h = R_LEDC_Step_Up (4:0) default 07h
PAJinit(152) = &H3507 ' addr 35h = R_LEDC_Step_Down (4:0) default 07h
PAJinit(153) = &H3601 ' addr 36h = R_I2C_AutoResync_EnH default 01h
PAJinit(154) = &H37FF ' addr 37h = R_I2C_AutoResync_Time (7:0) default FFh
PAJinit(155) = &H3836 ' addr 38h = R_adc_control default 03h
PAJinit(156) = &H3907 ' addr 39h = R_pwrrsv_control default 01h
PAJinit(157) = &H3A00 ' addr 3Ah = R_I2CID_Sel (2:0) default 00h
PAJinit(158) = &H3EFF ' addr 3Eh = Cmd_DebugPettern (7:0) default 00h
PAJinit(159) = &H3F00 ' addr 3Fh = Cmd_DebugPattern (8) default 00h
PAJinit(160) = &H4077 ' addr 40h = R_cds_mode  default 00h
PAJinit(161) = &H4140 ' addr 41h = R_dac_control default 00h
PAJinit(162) = &H4200 ' addr 42h = R_global (3:0) default 08h
PAJinit(163) = &H4330 ' addr 43h = R_pga_test  default 00h
PAJinit(164) = &H44A0 ' addr 44h = R_ggh (1:0) default 00h
PAJinit(165) = &H455C ' addr 45h = R_adc_test  default 00h
PAJinit(166) = &H4600 ' addr 46h = R_ScanDAC_Control  default 00h
PAJinit(167) = &H4700 ' addr 47h = R_dac_control default 00h
PAJinit(168) = &H4858 ' addr 48h = R_PDN_Test default 01h
PAJinit(169) = &H4A1E ' addr 4Ah = R_CP_WOI_HSize (5:0) default 1Eh
PAJinit(170) = &H4B1E ' addr 4Bh = R_CP_WOI_VSize (5:0) default 1Eh
PAJinit(171) = &H4C00 ' addr 4Ch = R_CP_WOI_HStart (5:0) default 00h
PAJinit(172) = &H4D00 ' addr 4Dh = R_CP_WOI_VStart (5:0) default 00h
PAJinit(173) = &H4EA0 ' addr 4Eh = R_Clamp_control  default A0h
PAJinit(174) = &H4F80 ' addr 4Fh = R_WAKE_ivreg_EnL (7:0)  default 80h
PAJinit(175) = &H5000 ' addr 50h = R_bgp_control  default 00h
PAJinit(176) = &H5100 ' addr 51h = R_TS_Test  default 00h
PAJinit(177) = &H5200 ' addr 52h = R_ref_gen_EnL_Manual  default 00h
PAJinit(178) = &H5300 ' addr 53h = R_ref_gen_Enl  default 00h
PAJinit(179) = &H5400 ' addr 54h = R_ref_gen_forceNoPwrsv_EnH  default 00h
PAJinit(180) = &H5780 ' addr 57h = R_WAKE_tg_EnL(7:0) default 80h
PAJinit(181) = &H5910 ' addr 59h = T_ab_blk_EnH  default 01h
PAJinit(182) = &H5A08 ' addr 5Ah = T_cds_asout_Enh  default 00h
PAJinit(183) = &H5B94 ' addr 5Bh = T_pixbias (2:0) default 04h
PAJinit(184) = &H5CE8 ' addr 5Ch = T_pxoset_EnH  default 01h
PAJinit(185) = &H5D08 ' addr 5Dh = T_tg_allow_EnH  default 00h
PAJinit(186) = &H5E3D ' addr 5Eh = T_clamp_drv_ctl (1:0) default 01h
PAJinit(187) = &H5F99 ' addr 5Fh = T_vdda28_lvl (2:0)  default 04h
PAJinit(188) = &H6045 ' addr 60h = T_vrtb_shift (1:0) default 01h
PAJinit(189) = &H6140 ' addr 61h = T_flush_lvl (2:0) default 04h
PAJinit(190) = &H632D ' addr 63h = T_adc_vcmtest_Enh  default 00h
PAJinit(191) = &H6402 ' addr 64h = T_GPIO_OPDRV (1:0) default 00h
PAJinit(192) = &H6596 ' addr 65h = R_IDLE_TIME (7:0) default B4h
PAJinit(193) = &H6600 ' addr 66h = R_IDLE_TIME (15:8) default B4h
PAJinit(194) = &H6797 ' addr 67h = R_IDLE_TIME_SLEEP_1 (7:0) default 168h (?)
PAJinit(195) = &H6801 ' addr 68h = R_IDLE_TIME_SLEEP_1 (15:8)
PAJinit(196) = &H69CD ' addr 69h = R_IDLE_TIME_SLEEP_2 (7:0) default 2D0h (?)
PAJinit(197) = &H6A01 ' addr 6Ah = R_IDLE_TIME_SLEEP_2 (15:8)
PAJinit(198) = &H6BB0 ' addr 6Bh = R_Obj_TIME_1 (7:0) default 4B0h (?)
PAJinit(199) = &H6C04 ' addr 6Ch = R_Obj_TIME_1 (15:8)
PAJinit(200) = &H6D2C ' addr 6Dh = R_Obj_TIME_2 (7:0) default 960h (?)
PAJinit(201) = &H6E01 ' addr 6Eh = R_Obj)TIME_2 (15:8)
PAJinit(202) = &H6F32 ' addr 6Fh = R_TG_INIT_TIME (7:0) default 32h
PAJinit(203) = &H7100 ' addr 71h = R_TG_POWERON_WAKEUP_TIME  default 00h
PAJinit(204) = &H7201 ' addr 72h = R_TG_EnH     default 00h
PAJinit(205) = &H7335 ' addr 73h = R_AUTO_SLEEP_Mode1/Mode2/Wakeup_Mode  default 00h
PAJinit(206) = &H7400 ' addr 74h = R_WakeUpSig_Sel (1:0) default 00h
PAJinit(207) = &H7533 ' addr 75h = T_ckt_test0  default 00h
PAJinit(208) = &H7631 ' addr 76h = T_I2C_OPDRV (1:0) default 01h
PAJinit(209) = &H7701 ' addr 77h = R_SRAM_Read_EnH default 00h
PAJinit(210) = &H7C84 ' addr 7Ch = R_SPIOUT_PXDNUM (7:0) default 384h (?)
PAJinit(211) = &H7D03 ' addr 7Dh = R_SPIOUT_PXDNUM (15:8)
PAJinit(212) = &H7E01 ' addr 7Eh = R_SPIOUT_EnH    default 00h
PAJinit(213) = &HEF01 ' send 01h TO ADDR EFh, to SWITCH TO BANK 1 <<*********
PAJinit(214) = &H6512 ' per Arduino code: near mode 240 fps
PAJinit(215) = &HEF00 ' send 00h TO ADDR EFh, to SWITCH TO BANK 0 <<*********
END SUB
' ******************************************************************